import { supabase, supabaseAdmin, TABLES } from '@/config/supabase';
import { 
  ConsultationForm, 
  ConsultationQuestion, 
  ConsultationResponse, 
  CreateFormRequest, 
  SubmitResponseRequest,
  FormWithQuestions,
  ResponseWithCustomer,
  ConsultationAnalytics
} from './types';
import { aiService } from '@/modules/ai/service';

export const consultationService = {
  // Create consultation form with questions
  async createForm(vendorId: string, formData: CreateFormRequest): Promise<{ data?: ConsultationForm; error?: string }> {
    try {
      const { title, description, is_paid, price, currency, fields } = formData;
      
      // Insert form
      const { data: form, error: formError } = await supabaseAdmin
        .from(TABLES.CONSULTATION_FORMS)
        .insert({
          vendor_id: vendorId,
          title,
          description,
          is_paid,
          price,
          currency,
          status: 'active'
        })
        .select()
        .single();

      if (formError) throw formError;

      // Insert questions
      const questions = fields.map((field, index) => ({
        form_id: form.id,
        label: field.label,
        type: field.type,
        options: field.options,
        required: field.required,
        order_index: field.order_index || index
      }));

      const { error: questionsError } = await supabaseAdmin
        .from(TABLES.CONSULTATION_QUESTIONS)
        .insert(questions);

      if (questionsError) throw questionsError;

      return { data: form };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Get forms for vendor
  async getVendorForms(vendorId: string): Promise<{ data?: ConsultationForm[]; error?: string }> {
    try {
      const { data, error } = await supabaseAdmin
        .from(TABLES.CONSULTATION_FORMS)
        .select('*')
        .eq('vendor_id', vendorId)
        .order('created_at', { ascending: false });

      if (error) throw error;
      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Get form with questions
  async getFormWithQuestions(formId: string): Promise<{ data?: FormWithQuestions; error?: string }> {
    try {
      // Get form
      const { data: form, error: formError } = await supabaseAdmin
        .from(TABLES.CONSULTATION_FORMS)
        .select('*')
        .eq('id', formId)
        .single();

      if (formError) throw formError;

      // Get questions
      const { data: questions, error: questionsError } = await supabaseAdmin
        .from(TABLES.CONSULTATION_QUESTIONS)
        .select('*')
        .eq('form_id', formId)
        .order('order_index', { ascending: true });

      if (questionsError) throw questionsError;

      return { data: { ...form, questions } };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Submit form response
  async submitResponse(formId: string, responseData: SubmitResponseRequest): Promise<{ data?: ConsultationResponse; error?: string }> {
    try {
      const { customer_id, invoice_id, order_id, responses } = responseData;

      // Validate form exists and is active
      const { data: form, error: formError } = await supabaseAdmin
        .from(TABLES.CONSULTATION_FORMS)
        .select('status, is_paid, price')
        .eq('id', formId)
        .single();

      if (formError) throw new Error('Form not found');
      if (form.status !== 'active') throw new Error('Form is not active');

      // Insert response
      const { data: response, error: responseError } = await supabaseAdmin
        .from(TABLES.CONSULTATION_RESPONSES)
        .insert({
          form_id: formId,
          customer_id,
          invoice_id,
          order_id,
          responses,
          submitted_at: new Date().toISOString()
        })
        .select()
        .single();

      if (responseError) throw responseError;

      // Generate AI summary if AI service is available
      try {
        const summary = await this.generateResponseSummary(responses);
        // Could store summary in a separate field or table
        console.log('AI Summary:', summary);
      } catch (aiError) {
        console.log('AI summary generation failed:', aiError);
      }

      return { data: response };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Get form responses
  async getFormResponses(formId: string, filters?: { customer_id?: string; date_range?: string }): Promise<{ data?: ResponseWithCustomer[]; error?: string }> {
    try {
      let query = supabaseAdmin
        .from(TABLES.CONSULTATION_RESPONSES)
        .select(`
          *,
          customers:customer_id(id, name, email)
        `)
        .eq('form_id', formId);

      if (filters?.customer_id) {
        query = query.eq('customer_id', filters.customer_id);
      }

      if (filters?.date_range) {
        const [start, end] = filters.date_range.split(' to ');
        query = query.gte('created_at', start).lte('created_at', end);
      }

      const { data, error } = await query.order('created_at', { ascending: false });

      if (error) throw error;

      const responsesWithCustomer = data?.map(response => ({
        ...response,
        customer: response.customers
      }));

      return { data: responsesWithCustomer };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Get consultation analytics
  async getFormAnalytics(formId: string): Promise<{ data?: ConsultationAnalytics; error?: string }> {
    try {
      const { data: responses, error } = await supabaseAdmin
        .from(TABLES.CONSULTATION_RESPONSES)
        .select('*')
        .eq('form_id', formId);

      if (error) throw error;

      if (!responses || responses.length === 0) {
        return { data: {
          total_responses: 0,
          average_completion_time: 0,
          most_common_answers: {},
          popular_questions: []
        }};
      }

      // Calculate analytics
      const totalResponses = responses.length;
      const mostCommonAnswers: Record<string, any> = {};
      const questionCounts: Record<string, number> = {};

      responses.forEach(response => {
        response.responses.forEach((resp: any) => {
          const questionId = resp.question_id;
          const answer = resp.answer;
          
          if (!mostCommonAnswers[questionId]) {
            mostCommonAnswers[questionId] = {};
          }
          
          if (!mostCommonAnswers[questionId][answer]) {
            mostCommonAnswers[questionId][answer] = 0;
          }
          
          mostCommonAnswers[questionId][answer]++;
          
          questionCounts[questionId] = (questionCounts[questionId] || 0) + 1;
        });
      });

      // Get popular questions (most answered)
      const popularQuestions = Object.entries(questionCounts)
        .sort(([,a], [,b]) => b - a)
        .slice(0, 5)
        .map(([questionId]) => questionId);

      return { data: {
        total_responses: totalResponses,
        average_completion_time: 0, // Would need timing data
        most_common_answers: mostCommonAnswers,
        popular_questions: popularQuestions
      }};
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Generate AI summary of responses
  async generateResponseSummary(responses: any[]): Promise<string> {
    try {
      const responseText = responses.map(r => `${r.question_id}: ${r.answer}`).join('\n');
      const prompt = `Analyze this consultation form response and provide a brief summary for the vendor:\n\n${responseText}\n\nSummary:`;
      
      return await aiService.generateGeneralAssistant(prompt, 'consultation_analysis');
    } catch (error) {
      throw new Error('Failed to generate AI summary');
    }
  },

  // Duplicate form
  async duplicateForm(vendorId: string, formId: string, newTitle: string): Promise<{ data?: ConsultationForm; error?: string }> {
    try {
      const { data: originalForm, error: formError } = await this.getFormWithQuestions(formId);
      if (formError) throw new Error(formError);

      if (!originalForm) throw new Error('Form not found');

      // Create new form
      const { data: newForm, error: createError } = await this.createForm(vendorId, {
        title: newTitle,
        description: originalForm.description,
        is_paid: originalForm.is_paid,
        price: originalForm.price,
        currency: originalForm.currency,
        fields: originalForm.questions!.map(q => ({
          label: q.label,
          type: q.type,
          options: q.options,
          required: q.required,
          order_index: q.order_index
        }))
      });

      if (createError) throw new Error(createError);

      return { data: newForm };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // Update form status
  async updateFormStatus(formId: string, status: 'draft' | 'active' | 'inactive'): Promise<{ data?: ConsultationForm; error?: string }> {
    try {
      const { data, error } = await supabaseAdmin
        .from(TABLES.CONSULTATION_FORMS)
        .update({ status })
        .eq('id', formId)
        .select()
        .single();

      if (error) throw error;
      return { data };
    } catch (error: any) {
      return { error: error.message };
    }
  }
}; 